// Studio Rental Website - Version 4 (HTML/CSS/JS)

const studioTypes = [
  { 
    id: 'backdrop', 
    name: 'Студия с фоном', 
    icon: 'icons/type-backdrop.svg',
    description: 'Профессиональные фоны и освещение'
  },
  { 
    id: 'chromakey', 
    name: 'Хромакей', 
    icon: 'icons/type-chromakey.svg',
    description: 'Зеленая стена для видеосъемки'
  },
  { 
    id: 'interior', 
    name: 'Интерьер', 
    icon: 'icons/type-interior.svg',
    description: 'Уютные интерьерные пространства'
  },
  { 
    id: 'openspace', 
    name: 'Open Space', 
    icon: 'icons/type-openspace.svg',
    description: 'Большое открытое пространство'
  }
]

const studios = [
  { name: 'StudioSpace Premium', location: 'Центр', rating: 5, types: 'Все типы' },
  { name: 'StudioSpace Chroma', location: 'Тверская', rating: 5, types: 'Хромакей, Фон' },
  { name: 'StudioSpace Interior', location: 'Арбат', rating: 5, types: 'Интерьер, Open Space' },
  { name: 'StudioSpace Creative', location: 'Сокольники', rating: 5, types: 'Все типы' }
]

const guarantees = [
  { icon: 'icons/stat-bookings.svg', title: 'Профессиональное оборудование', text: 'Лучшая техника' },
  { icon: 'icons/stat-studios.svg', title: 'Быстрое бронирование', text: 'За минуты' },
  { icon: 'icons/stat-equipment.svg', title: 'Гибкий график', text: 'Работаем 24/7' },
  { icon: 'icons/stat-rating.svg', title: 'Высокий рейтинг', text: '98% довольных клиентов' }
]

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadBookings()
  updateStats()
  
  setInterval(() => {
    loadBookings()
    updateStats()
  }, 500)
}

function loadBookings() {
  const saved = localStorage.getItem('studioBookings')
  let bookings = []
  
  if (saved) {
    try {
      bookings = JSON.parse(saved)
    } catch (e) {
      bookings = []
    }
  } else {
    bookings = [
      {
        id: 1,
        studio: 'StudioSpace Premium',
        date: '2025-03-20',
        equipment: 'Профессиональное освещение, камеры',
        review: 'Отличная студия, все оборудование работает идеально!'
      },
      {
        id: 2,
        studio: 'StudioSpace Chroma',
        date: '2025-03-15',
        equipment: 'Хромакей, студийное освещение',
        review: 'Прекрасные условия для съемки'
      },
      {
        id: 3,
        studio: 'StudioSpace Interior',
        date: '2025-03-10',
        equipment: 'Интерьерная студия, естественный свет',
        review: 'Очень уютная атмосфера'
      }
    ]
    localStorage.setItem('studioBookings', JSON.stringify(bookings))
  }
  
  renderBookings(bookings)
  return bookings
}

function renderBookings(bookings) {
  const bookingsVertical = document.getElementById('bookingsVertical')
  if (!bookingsVertical) return
  
  if (bookings.length === 0) {
    bookingsVertical.innerHTML = `
      <div class="empty-state">
        <img src="icons/logo-camera.svg" alt="" class="empty-icon" width="64" height="64">
        <h3 class="empty-title">Нет бронирований</h3>
        <p class="empty-text">Забронируйте студию для вашего проекта</p>
      </div>
    `
  } else {
    bookingsVertical.innerHTML = bookings.map(booking => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      
      return `
        <div class="booking-item">
          <div class="booking-item-left">
            <div class="booking-item-icon">
              <img src="icons/booking-video.svg" alt="Camera" width="24" height="24">
            </div>
            <div class="booking-item-content">
              <div class="booking-item-header">
                <h3 class="booking-item-studio">${booking.studio}</h3>
                <div class="booking-item-rating">
                  ${stars}
                </div>
              </div>
              <div class="booking-item-details">
                <div class="booking-item-detail">
                  <img src="icons/booking-calendar.svg" alt="Calendar" width="16" height="16">
                  <span>
                    ${new Date(booking.date).toLocaleDateString('ru-RU', { 
                      day: 'numeric', 
                      month: 'long', 
                      year: 'numeric' 
                    })}
                  </span>
                </div>
                <div class="booking-item-detail">
                  <img src="icons/booking-video.svg" alt="Video" width="16" height="16">
                  <span>${booking.equipment}</span>
                </div>
                ${booking.review ? `
                  <div class="booking-item-review">
                    <img src="icons/booking-message.svg" alt="Message" width="16" height="16">
                    <span>${booking.review}</span>
                  </div>
                ` : ''}
              </div>
            </div>
          </div>
        </div>
      `
    }).join('')
  }
}

function updateStats() {
  const bookings = loadBookings()
  const stats = [
    { 
      value: bookings.length, 
      label: 'Бронирований'
    },
    { 
      value: 4.8, 
      label: 'Рейтинг'
    },
    { 
      value: new Set(bookings.map(b => b.studio)).size, 
      label: 'Студий'
    }
  ]
  
  const statsMinimal = document.getElementById('statsMinimal')
  if (statsMinimal) {
    statsMinimal.innerHTML = stats.map(stat => `
      <div class="stat-minimal-item">
        <div class="stat-minimal-value">${stat.value}</div>
        <div class="stat-minimal-label">${stat.label}</div>
      </div>
    `).join('')
  }
}

function initBookPage() {
  const bookingForm = document.getElementById('bookingForm')
  const typesList = document.getElementById('typesList')
  const dateInput = document.getElementById('date')
  const successMessage = document.getElementById('successMessage')
  const guaranteesGrid = document.getElementById('guaranteesGrid')
  const studiosList = document.getElementById('studiosList')

  let selectedType = null

  if (typesList) {
    typesList.innerHTML = studioTypes.map(type => `
      <button type="button" class="type-item" data-id="${type.id}">
        <div class="type-item-icon">
          <img src="${type.icon}" alt="${type.name}" width="24" height="24">
        </div>
        <div class="type-item-content">
          <h3 class="type-item-name">${type.name}</h3>
          <p class="type-item-desc">${type.description}</p>
        </div>
      </button>
    `).join('')

    typesList.querySelectorAll('.type-item').forEach(button => {
      button.addEventListener('click', function() {
        typesList.querySelectorAll('.type-item').forEach(btn => {
          btn.classList.remove('selected')
        })
        this.classList.add('selected')
        selectedType = studioTypes.find(t => t.id === this.dataset.id)
      })
    })
  }

  if (guaranteesGrid) {
    guaranteesGrid.innerHTML = guarantees.map(guarantee => `
      <div class="guarantee-item">
        <div class="guarantee-icon">
          <img src="${guarantee.icon}" alt="" width="24" height="24">
        </div>
        <div class="guarantee-content">
          <h3 class="guarantee-title">${guarantee.title}</h3>
          <p class="guarantee-text">${guarantee.text}</p>
        </div>
      </div>
    `).join('')
  }

  if (studiosList) {
    studiosList.innerHTML = studios.map(studio => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      
      return `
        <div class="studio-item">
          <div class="studio-item-left">
            <div class="studio-item-icon">
              <img src="icons/studio-camera.svg" alt="" width="24" height="24">
            </div>
            <div class="studio-item-content">
              <h3 class="studio-item-name">${studio.name}</h3>
              <div class="studio-item-details">
                <span class="studio-item-location">${studio.location}</span>
                <div class="studio-item-rating">
                  ${stars}
                </div>
              </div>
              <p class="studio-item-types">${studio.types}</p>
            </div>
          </div>
        </div>
      `
    }).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (bookingForm) {
    bookingForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedType && dateInput.value) {
        const savedBookings = localStorage.getItem('studioBookings')
        const bookings = savedBookings ? JSON.parse(savedBookings) : []
        
        const randomStudio = studios[Math.floor(Math.random() * studios.length)]
        const newBooking = {
          id: Date.now(),
          studio: randomStudio.name,
          date: dateInput.value,
          equipment: selectedType.description,
          review: ''
        }

        bookings.push(newBooking)
        localStorage.setItem('studioBookings', JSON.stringify(bookings))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedType = null
          dateInput.value = ''
          typesList.querySelectorAll('.type-item').forEach(btn => {
            btn.classList.remove('selected')
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  if (slides.length === 0) return
  
  slides[currentSlide].classList.remove('active')
  if (dots[currentSlide]) dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  if (dots[currentSlide]) dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

